package ch.frankel.blog.hibernate;

import static org.testng.Assert.assertSame;

import org.hibernate.Criteria;
import org.hibernate.classic.Session;
import org.hibernate.criterion.Projections;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

import ch.frankel.blog.hibernate.model.Product;

/**
 * Test to check logical deletion management.
 * 
 * @author Nicolas Frnkel
 * @since 19 fvr. 2010
 */
public class TestPart5 extends AbstractTest {

    /**
     * Opens transaction.
     */
    @BeforeClass(dependsOnMethods = "setUpFactory")
    protected void beginTransaction() {

        factory.getCurrentSession().beginTransaction();
    }

    /**
     * Test the count with no filter enabled.
     */
    @Test
    public void testCountAll() {

        int count = (Integer) factory.getCurrentSession().createCriteria(Product.class).setProjection(
                Projections.rowCount()).uniqueResult();

        assertSame(count, 4);
    }

    /**
     * Test the count with filter enabled.
     */
    @Test
    public void testCountFiltered() {

        Session session = factory.getCurrentSession();

        session.enableFilter("activeProducts");

        int count = (Integer) session.createCriteria(Product.class).setProjection(Projections.rowCount())
                .uniqueResult();

        assertSame(count, 2);

        session.disableFilter("activeProducts");
    }

    /**
     * Test logical deletion.
     */
    @Test(dependsOnMethods = { "testCountAll", "testCountFiltered" })
    public void testLogicalDeletion() {

        Session session = factory.getCurrentSession();

        Product product = (Product) session.get(Product.class, 2);

        session.delete(product);

        // Synchronize with DB
        session.flush();

        Criteria projection = session.createCriteria(Product.class).setProjection(Projections.rowCount());

        int count = (Integer) projection.uniqueResult();

        // Assert that there are still 4 records
        assertSame(count, 4);

        session.enableFilter("activeProducts");

        count = (Integer) projection.uniqueResult();

        // Assert that there only one still active
        assertSame(count, 1);
    }

    /**
     * Rollbacks transaction.
     */
    @AfterClass
    protected void rollback() {

        factory.getCurrentSession().getTransaction().rollback();
    }
}
